/* implementacija hijerarhije je analogna onoj iz primera sa nasledjivanjem
 * i nece biti posebno komentarisana 
 */
#include "Funkcije.hpp"

std::ostream& operator <<(std::ostream& s, const Funkcija& f) {
    f.Print(s);
    return s;
}

KonstantnaFunkcija::KonstantnaFunkcija(double val) {
    _value = val;
}

KonstantnaFunkcija::KonstantnaFunkcija(const KonstantnaFunkcija& kf) {
    _value = kf._value;
}

void KonstantnaFunkcija::Print(std::ostream& s) const {
    s << _value;
}

Funkcija* KonstantnaFunkcija::Clone() const {
    
    return new KonstantnaFunkcija(*this);
}

double KonstantnaFunkcija::Value(double x) const {
    
    return _value;
}

Funkcija* KonstantnaFunkcija::Diff() const {

    return new KonstantnaFunkcija(0);
}

Funkcija* KonstantnaFunkcija::Compose(Funkcija* f) const {

    return Clone();
}

void IdentitetFunkcija::Print(std::ostream& s) const {
    s << "x";
}

Funkcija* IdentitetFunkcija::Clone() const {
    return new IdentitetFunkcija(*this);
}

double IdentitetFunkcija::Value(double x) const {

    return x;
}

Funkcija* IdentitetFunkcija::Diff() const {

    return new KonstantnaFunkcija(1);
}

Funkcija* IdentitetFunkcija::Compose(Funkcija* f) const {

    return f->Clone();
}

BinarniOperator::BinarniOperator(const std::string& s, Funkcija* l, Funkcija* r) 
    : _symbol(s), _left(l), _right(r) {

    }
BinarniOperator::~BinarniOperator() {
    delete _left;
    delete _right;
}

void BinarniOperator::Print(std::ostream& s) const {

    s << (*_left) << " " << _symbol << " " << (*_right);
}


PlusFunkcija::PlusFunkcija(Funkcija* l, Funkcija* d) 
    : BinarniOperator("+", l, d) {

    }
PlusFunkcija::PlusFunkcija(const PlusFunkcija& pf)
    : BinarniOperator("+", pf._left->Clone(), pf._right->Clone()) {

    }

Funkcija* PlusFunkcija::Clone() const {
    return new PlusFunkcija(*this);
}
double PlusFunkcija::Value(double x) const {
    return _left->Value(x) + _right->Value(x);
}
Funkcija* PlusFunkcija::Diff() const {
    return new PlusFunkcija(_left->Diff(), _right->Diff());
}
Funkcija* PlusFunkcija::Compose(Funkcija* f) const {
    return new PlusFunkcija(_left->Compose(f), _right->Compose(f));
}

MinusFunkcija::MinusFunkcija(Funkcija* l, Funkcija* d)
    : BinarniOperator("-", l, d) {
        
    }
MinusFunkcija::MinusFunkcija(const MinusFunkcija& pf)
    : BinarniOperator("-", pf._left->Clone(), pf._right->Clone()) {
        
    }

Funkcija* MinusFunkcija::Clone() const {
    return new MinusFunkcija(*this);
}
double MinusFunkcija::Value(double x) const{
    return _left->Value(x) - _right->Value(x);
}
Funkcija* MinusFunkcija::Diff() const {
    return new MinusFunkcija(_left->Diff(), _right->Diff());
}
Funkcija* MinusFunkcija::Compose(Funkcija* f) const {
    return new MinusFunkcija(_left->Compose(f), _right->Compose(f));
}

PutaFunkcija::PutaFunkcija(Funkcija* l, Funkcija* d)
    : BinarniOperator("*", l, d) {
        
    }
PutaFunkcija::PutaFunkcija(const PutaFunkcija& pf)
    : BinarniOperator("*", pf._left->Clone(), pf._right->Clone()) {
        
    }

Funkcija* PutaFunkcija::Clone() const {
    return new PutaFunkcija(*this);
}
double PutaFunkcija::Value(double x) const {
    return _left->Value(x) * _right->Value(x);
}
Funkcija* PutaFunkcija::Diff() const {
    return new PlusFunkcija(
                    new PutaFunkcija(_left->Diff(), _right->Clone()),
                    new PutaFunkcija(_left->Clone(), _right->Diff())
                );
}
Funkcija* PutaFunkcija::Compose(Funkcija* f) const {
    return new PutaFunkcija(_left->Compose(f), _right->Compose(f));
}

PodeljenoFunkcija::PodeljenoFunkcija(Funkcija* l, Funkcija* d)
    : BinarniOperator("/", l, d) {
        
    }
PodeljenoFunkcija::PodeljenoFunkcija(const PodeljenoFunkcija& pf)
    : BinarniOperator("/", pf._left->Clone(), pf._right->Clone()) {
        
    }

Funkcija* PodeljenoFunkcija::Clone() const {
    return new PodeljenoFunkcija(*this);
}
double PodeljenoFunkcija::Value(double x) const {
    return _left->Value(x) / _right->Value(x);
}
Funkcija* PodeljenoFunkcija::Diff() const {

    return new PodeljenoFunkcija(
                    new MinusFunkcija(
                        new PutaFunkcija(_left->Diff(), _right->Clone()),
                        new PutaFunkcija(_left->Clone(), _right->Diff())
                    ),
                    new PutaFunkcija(_right->Clone(), _right->Clone())
                );
}
Funkcija* PodeljenoFunkcija::Compose(Funkcija* f) const {
    
    return new PodeljenoFunkcija(_left->Compose(f), _right->Compose(f));
}


ElementarnaFunkcija::ElementarnaFunkcija(const std::string& s, Funkcija* f) :
    _name(s), _expr(f) {

    }
ElementarnaFunkcija::~ElementarnaFunkcija() {

    delete _expr;
}

void ElementarnaFunkcija::Print(std::ostream& s) const {

    s << _name << "(" << (*_expr) << ")";
}

SuprotnaFunkcija::SuprotnaFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("-", expr) {

    }
SuprotnaFunkcija::SuprotnaFunkcija(const SuprotnaFunkcija& sf)
    : ElementarnaFunkcija("-", sf._expr->Clone()) {

    }

Funkcija* SuprotnaFunkcija::Clone() const {
    return new SuprotnaFunkcija(*this);
}
double SuprotnaFunkcija::Value(double x) const {
    return -_expr->Value(x);
}
Funkcija* SuprotnaFunkcija::Diff() const {
    return new SuprotnaFunkcija(_expr->Diff());
}
Funkcija* SuprotnaFunkcija::Compose(Funkcija* f) const {
    return new SuprotnaFunkcija(_expr->Compose(f));
}

SinFunkcija::SinFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("sin", expr) {

    }
SinFunkcija::SinFunkcija(const SinFunkcija& sf)
    : ElementarnaFunkcija("sin", sf._expr->Clone()) {

    }

Funkcija* SinFunkcija::Clone() const {
    return new SinFunkcija(*this);
}
double SinFunkcija::Value(double x) const {
    return sin(_expr->Value(x));
}
Funkcija* SinFunkcija::Diff() const {
    return new PutaFunkcija(_expr->Diff(), new CosFunkcija(_expr->Clone()));
}
Funkcija* SinFunkcija::Compose(Funkcija* f) const {
    return new SinFunkcija(_expr->Compose(f));
}

CosFunkcija::CosFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("cos", expr) {

    }
CosFunkcija::CosFunkcija(const CosFunkcija& sf)
    : ElementarnaFunkcija("cos", sf._expr->Clone()) {

    }

Funkcija* CosFunkcija::Clone() const {
    return new CosFunkcija(*this);
}
double CosFunkcija::Value(double x) const {
    return cos(_expr->Value(x));
}
Funkcija* CosFunkcija::Diff() const {
    return new PutaFunkcija(
                new KonstantnaFunkcija(-1),
                new PutaFunkcija(_expr->Diff(), new SinFunkcija(_expr->Clone()))
                );
}
Funkcija* CosFunkcija::Compose(Funkcija* f) const {
    return new CosFunkcija(_expr->Compose(f));
}

TanFunkcija::TanFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("tan", expr) {

    }
TanFunkcija::TanFunkcija(const TanFunkcija& sf)
    : ElementarnaFunkcija("tan", sf._expr->Clone()) {

    }

Funkcija* TanFunkcija::Clone() const {
    return new TanFunkcija(*this);
}
double TanFunkcija::Value(double x) const {
    return tan(_expr->Value(x));
}
Funkcija* TanFunkcija::Diff() const {
    return new PodeljenoFunkcija(
            new KonstantnaFunkcija(1),
            new PutaFunkcija(new CosFunkcija(_expr->Clone()), new CosFunkcija(_expr->Clone()))
        );
}
Funkcija* TanFunkcija::Compose(Funkcija* f) const {
    return new TanFunkcija(_expr->Compose(f));
}

CotFunkcija::CotFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("cot", expr) {

    }
CotFunkcija::CotFunkcija(const CotFunkcija& sf)
    : ElementarnaFunkcija("cot", sf._expr->Clone()) {

    }

Funkcija* CotFunkcija::Clone() const {
    return new CotFunkcija(*this);
}
double CotFunkcija::Value(double x) const {
    return 1/tan(_expr->Value(x));
}
Funkcija* CotFunkcija::Diff() const {
    return new PodeljenoFunkcija(
            new KonstantnaFunkcija(-1),
            new PutaFunkcija(new SinFunkcija(_expr->Clone()), new SinFunkcija(_expr->Clone()))
        );
}
Funkcija* CotFunkcija::Compose(Funkcija* f) const {
    return new CotFunkcija(_expr->Compose(f));
}

ExpFunkcija::ExpFunkcija(Funkcija* expr)    
    : ElementarnaFunkcija("exp", expr) {

    }
ExpFunkcija::ExpFunkcija(const ExpFunkcija& sf)
    : ElementarnaFunkcija("exp", sf._expr->Clone()) {

    }

Funkcija* ExpFunkcija::Clone() const {
    return new ExpFunkcija(*this);
}
double ExpFunkcija::Value(double x) const {
    return exp(_expr->Value(x));
}
Funkcija* ExpFunkcija::Diff() const {
    return new PutaFunkcija(_expr->Diff(), new ExpFunkcija(_expr->Clone()));
}
Funkcija* ExpFunkcija::Compose(Funkcija* f) const {
    return new ExpFunkcija(_expr->Compose(f));
}

LogFunkcija::LogFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("log", expr) {

    }
LogFunkcija::LogFunkcija(const LogFunkcija& sf)
    : ElementarnaFunkcija("log", sf._expr->Clone()) {

    }

Funkcija* LogFunkcija::Clone() const {
    return new LogFunkcija(*this);
}
double LogFunkcija::Value(double x) const {
    return log(_expr->Value(x));
}
Funkcija* LogFunkcija::Diff() const {
    return new PodeljenoFunkcija(
        _expr->Diff(),
        new LogFunkcija(_expr->Clone())
    );
}
Funkcija* LogFunkcija::Compose(Funkcija* f) const {

    return new LogFunkcija(_expr->Compose(f));
}

PowFunkcija::PowFunkcija(Funkcija* expr, double step)
    : ElementarnaFunkcija("pow", expr) {
        _stepen = step;
    }
PowFunkcija::PowFunkcija(const PowFunkcija& sf)
    : ElementarnaFunkcija("pow", sf._expr->Clone()) {
        _stepen = sf._stepen;
    }

Funkcija* PowFunkcija::Clone() const {
    return new PowFunkcija(*this);
}
double PowFunkcija::Value(double x) const {
    return pow(_expr->Value(x), _stepen);
}
Funkcija* PowFunkcija::Diff() const {
    return new PutaFunkcija(
        new PutaFunkcija(new KonstantnaFunkcija(_stepen), _expr->Diff()),
        new PowFunkcija(_expr->Clone(), _stepen-1)
    );
}
Funkcija* PowFunkcija::Compose(Funkcija* f) const {

    return new PowFunkcija(_expr->Compose(f), _stepen);
}

SqrtFunkcija::SqrtFunkcija(Funkcija* expr)
    : ElementarnaFunkcija("sqrt", expr) {

    }
SqrtFunkcija::SqrtFunkcija(const SqrtFunkcija& sf)
    : ElementarnaFunkcija("sqrt", sf._expr->Clone()) {

    }

Funkcija* SqrtFunkcija::Clone() const {
    return new SqrtFunkcija(*this);
}
double SqrtFunkcija::Value(double x) const {
    return sqrt(_expr->Value(x));
}
Funkcija* SqrtFunkcija::Diff() const {
    return new PodeljenoFunkcija(
        _expr->Diff(),
        new PutaFunkcija(new KonstantnaFunkcija(2), new SqrtFunkcija(_expr->Clone()))
    );
}
Funkcija* SqrtFunkcija::Compose(Funkcija* f) const {
    return new SqrtFunkcija(_expr->Compose(f));
}